<?php

Class WC_Tracking {
	protected static $instance;
	public $link = 'http://parcelsapp.com/en/tracking/';
	
	/**
	 * Cache for order objects to improve performance
	 * @var array
	 */
	protected static $order_cache = [];

	public function __construct() {
		if ( ! class_exists('Woocommerce') ) return;

		// Hook for both generic and specific screen IDs
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ), 10, 2 );
		
		// Hook for HPOS screen ID (if available)
		if( function_exists( 'wc_get_page_screen_id' ) ) {

			$screen_id = wc_get_page_screen_id( 'shop-order' );
			add_action( 'add_meta_boxes_' . $screen_id, array( $this, 'add_meta_box_hpos' ), 10, 1 );
		}
		
		// Hook for legacy shop_order post type
		add_action( 'add_meta_boxes_shop_order', array( $this, 'add_meta_box_legacy' ), 10, 1 );
		
		add_action( 'woocommerce_process_shop_order_meta', array( $this, 'save_meta_box' ), 9999 );
		
		// Clear order cache when order is updated
		add_action( 'woocommerce_update_order', array( $this, 'clear_order_cache_on_update' ), 1, 1 );
		add_action( 'woocommerce_new_order', array( $this, 'clear_order_cache_on_update' ), 1, 1 );
	}
	
	/**
	 * Clear order cache when order is updated
	 * 
	 * @param int $order_id Order ID
	 */
	public function clear_order_cache_on_update( $order_id ) {

		self::clear_order_cache( $order_id );
	}

	public static function get_instance()  {
        if ( !isset( self::$instance) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

	/**
	 * Get cached order object or fetch and cache it
	 * 
	 * @param int|WC_Order $order_id Order ID or WC_Order object
	 * @return WC_Order|null Order object or null if not found
	 */
	public function get_cached_order( $order_id ) {

		// If already a WC_Order object, return it
		if( is_a( $order_id, 'WC_Order' ) ) {

			return $order_id;
		}

		$order_id_value = intval( $order_id );

		// Check cache first
		if( isset( self::$order_cache[$order_id_value] ) ) {

			return self::$order_cache[$order_id_value];
		}

		// Fetch and cache
		$order = wc_get_order( $order_id_value );
		
		if( $order && is_a( $order, 'WC_Order' ) ) {

			self::$order_cache[$order_id_value] = $order;
		}

		return $order;
	}

	/**
	 * Clear cached order object
	 * 
	 * @param int $order_id Order ID
	 */
	public static function clear_order_cache( $order_id = null ) {

		if( $order_id !== null ) {

			unset( self::$order_cache[intval( $order_id )] );
		} else {

			self::$order_cache = [];
		}
	}

	/**
	 * Get order meta with HPOS and legacy fallback support
	 * 
	 * @param int|WC_Order $order_id Order ID or WC_Order object
	 * @param string $meta_key Meta key to retrieve
	 * @param bool $single Whether to return single value (default: true)
	 * @return mixed Meta value or empty string if not found
	 */
	public function get_meta( $order_id, $meta_key, $single = true ) {

		$order = $this->get_cached_order( $order_id );
		$order_id_value = null;

		if( $order && is_a( $order, 'WC_Order' ) ) {

			$order_id_value = $order->get_id();
		} else {

			$order_id_value = is_a( $order_id, 'WC_Order' ) ? $order_id->get_id() : intval( $order_id );
		}

		$meta_value = null;

		// Try order object method first (HPOS compatible)
		if( $order && is_a( $order, 'WC_Order' ) ) {

			$meta_value = $order->get_meta( $meta_key, $single );
		}

		// Fallback to get_post_meta for legacy or if order method returns empty
		if( $meta_value === null || $meta_value === '' || $meta_value === false ) {

			$meta_value = get_post_meta( $order_id_value, $meta_key, $single );
		}

		return $meta_value;
	}

	/**
	 * Update order meta with HPOS and legacy fallback support
	 * 
	 * @param int|WC_Order $order_id Order ID or WC_Order object
	 * @param string $meta_key Meta key to update
	 * @param mixed $meta_value Meta value to save
	 * @return bool|int True on success, meta ID on success, false on failure
	 */
	public function update_meta( $order_id, $meta_key, $meta_value ) {

		$order = $this->get_cached_order( $order_id );
		$order_id_value = null;

		if( $order && is_a( $order, 'WC_Order' ) ) {

			$order_id_value = $order->get_id();
		} else {

			$order_id_value = is_a( $order_id, 'WC_Order' ) ? $order_id->get_id() : intval( $order_id );
		}

		// Try order object method first (HPOS compatible)
		if( $order && is_a( $order, 'WC_Order' ) ) {

			$order->update_meta_data( $meta_key, $meta_value );
			$saved = $order->save();

			// If save succeeded, clear cache and return true
			if( $saved !== false ) {

				self::clear_order_cache( $order_id_value );
				return true;
			}
		}

		// Fallback to update_post_meta for legacy or if order save fails
		$result = update_post_meta( $order_id_value, $meta_key, $meta_value );
		
		// Clear cache after update
		if( $result !== false ) {
			self::clear_order_cache( $order_id_value );
		}
		
		return $result;
	}

	public function add_meta_box( $post_type = null, $post_or_order = null ) {

		// Get the current screen
		$screen = get_current_screen();
		
		// Determine the screen ID for orders
		$screen_id = 'shop_order';
		
		if( $screen && function_exists( 'wc_get_page_screen_id' ) ) {

			$screen_id = wc_get_page_screen_id( 'shop-order' );
		} elseif( $screen ) {

			$screen_id = $screen->id;
		}

		// Check if we're on an order screen
		$is_order_screen = false;
		
		if( $screen ) {

			// Check for HPOS screen
			if( strpos( $screen->id, 'wc-orders' ) !== false || strpos( $screen->id, 'shop-order' ) !== false ) {

				$is_order_screen = true;
			}
			
			// Check for legacy shop_order post type
			if( $screen->id === 'shop_order' || $post_type === 'shop_order' ) {

				$is_order_screen = true;
			}
		}

	if( ! $is_order_screen ) {

		return;
	}

		// Get order ID from post or order object
		$order_id = null;
		
		if( is_object( $post_or_order ) ) {

			if( $post_or_order instanceof WC_Order ) {

				$order_id = $post_or_order->get_id();
			} elseif( $post_or_order instanceof WP_Post ) {

				$order_id = $post_or_order->ID;
		}
	}

	// Add meta box for both HPOS and legacy
		add_meta_box( 
			'tee-tracking-number', 
			__( 'Order tracking', 'basr-core' ), 
			array( $this, 'show_order_tracking_metabox' ),
			$screen_id, 
			'side', 
			'high' 
		);
	}

	/**
	 * Add meta box for HPOS (High-Performance Order Storage)
	 */
	public function add_meta_box_hpos( $order ) {

		$screen = get_current_screen();
		$screen_id = $screen ? $screen->id : 'woocommerce_page_wc-orders';
		
		add_meta_box( 
			'tee-tracking-number', 
			__( 'Order tracking', 'basr-core' ), 
			array( $this, 'show_order_tracking_metabox' ),
			$screen_id, 
			'side', 
			'high' 
		);
	}

	/**
	 * Add meta box for legacy shop_order post type
	 */
	public function add_meta_box_legacy( $post ) {

		add_meta_box( 
			'tee-tracking-number', 
			__( 'Order tracking', 'basr-core' ), 
			array( $this, 'show_order_tracking_metabox' ),
			'shop_order', 
			'side', 
			'high' 
		);
	}

	function show_order_tracking_metabox( $post_or_order ) {
		
		// Handle both WP_Post and WC_Order objects
		$order_id = null;
		$order = null;
		
		if( $post_or_order instanceof WC_Order ) {

			$order_id = $post_or_order->get_id();
			$order = $post_or_order;
		} elseif( $post_or_order instanceof WP_Post ) {

			$order_id = $post_or_order->ID;
			$order = $this->get_cached_order( $order_id );
		} elseif( is_numeric( $post_or_order ) ) {

			$order_id = $post_or_order;
			$order = $this->get_cached_order( $order_id );
		} else {

			return;
		}

		if( ! $order_id || ! $order ) {

			return;
		}

		// Check for pending API updates and process them before displaying
		// This ensures the meta box shows the latest tracking numbers from API
		$new_trackings = tee_get_order_meta( $order, 'teeAllover_fulfill_2' );
		$last_processed = tee_get_order_meta( $order, '_tee_fulfillment_last_processed' );
		
		// If there's new API data that hasn't been processed, process it now
		// But only if we're not already processing (prevent loops)
		if( ! empty( $new_trackings ) && $new_trackings !== $last_processed ) {
			
			// Use a static flag to prevent recursive calls from meta box
			static $meta_box_processing = [];
			
			if( ! isset( $meta_box_processing[$order_id] ) ) {
				
				$meta_box_processing[$order_id] = true;
				
				// Clear cache to ensure fresh data
				self::clear_order_cache( $order_id );
				
				// Process the update synchronously to get latest data
				tee_fulfillment_woocommerce_order_update( $order_id );
				
				// Get fresh order object after processing
				$order = $this->get_cached_order( $order_id );
				
				unset( $meta_box_processing[$order_id] );
			}
		}

		// Use class method for HPOS and legacy compatibility
		$tee_tracking_number = $this->get_meta( $order, 'tee_tracking_number' );
		$order_tracking_link = $this->get_meta( $order, 'order_tracking_link' );
		$fulfill = $this->get_meta( $order, 'teeAllover_fulfill' );
		
		// Decode and validate tracking data
		if( ! empty( $fulfill ) && is_string( $fulfill ) ) {
			$decoded = json_decode( $fulfill, true );
			if( json_last_error() === JSON_ERROR_NONE ) {
				$fulfill = $decoded;
			} else {
				$fulfill = [];
			}
		} else {
			$fulfill = [];
		}
		
		// Convert objects to arrays and ensure it's a proper array
		$fulfill = tee_convert_obj_to_array( $fulfill );
		if( ! is_array( $fulfill ) ) {
			$fulfill = [];
		}

		// Remove duplicates and keep newest version of each tracking number
		if( is_array( $fulfill ) && ! empty( $fulfill ) ) {
			
			// Reverse array to process newest first, then deduplicate
			$fulfill = array_reverse( $fulfill );
			$seen_tracking_numbers = [];
			$deduplicated = [];
			
			foreach( $fulfill as $key => $tracking ) {
				
				$tracking_num = isset( $tracking['tracking_number'] ) ? trim( $tracking['tracking_number'] ) : 
				               ( isset( $tracking['tracking'] ) ? trim( $tracking['tracking'] ) : '' );
				
				// Only add if we haven't seen this tracking number yet (keep first/newest occurrence)
				if( ! empty( $tracking_num ) && ! isset( $seen_tracking_numbers[$tracking_num] ) ) {
					
					$seen_tracking_numbers[$tracking_num] = true;
					$deduplicated[] = $tracking;
				}
			}
			
			// Reverse back to show newest first
			$fulfill = array_reverse( $deduplicated );
		}

		// l( $fulfill );
		echo '<div class="tee-tracking">';

		if( is_array( $fulfill ) ) {
			foreach( $fulfill as $key => $tracking ) {	
				$this->print_tracking_html( $tracking );
			}
		}

		echo '</div>'; //.tee-tracking 

		echo '<div class="add-more"><span class="button">Add Tracking</span></div>';
		echo '<div class="mail-option"><input type="checkbox" name="tee-resend-tracking">Send Mail for Customer.</div>';
		echo '<input id="teeAllover_fulfill" name="teeAllover_fulfill" type="hidden" value="' . esc_attr( json_encode( $fulfill ) ) . '">';
		echo '<input id="tee_total_fulfill" name="tee_total_fulfill" type="hidden" value="' . esc_attr( $this->get_meta( $order, 'tee_total_fulfill' ) ) . '">';
		echo '<input id="tee_total_items" name="tee_total_items" type="hidden" value="' . esc_attr( $this->get_meta( $order, 'tee_total_items' ) ) . '">';
	}

	function print_tracking_html( $data ) {
		// Validate and convert data
		if( ! is_array( $data ) ) {
			$data = tee_convert_obj_to_array( $data );
		}
		
		if( ! is_array( $data ) ) {
			return; // Skip if data is invalid
		}
		
		// Ensure required fields exist with defaults
		$tracking_number = isset( $data['tracking_number'] ) ? $data['tracking_number'] : ( isset( $data['tracking'] ) ? $data['tracking'] : '' );
		$tracking_url = isset( $data['tracking_url'] ) ? $data['tracking_url'] : '';
		$line_items = isset( $data['line_items'] ) ? $data['line_items'] : [];
		
		if( ! is_array( $line_items ) ) {
			$line_items = tee_convert_obj_to_array( $line_items );
		}
		if( ! is_array( $line_items ) ) {
			$line_items = [];
		}
		
		?>
		<div class="track-information">
			<p><span class="notice-dismiss del-tracking"></span></p>
			<p>
				<label
					for="tee_tracking_number"> <?php _e ( 'Tracking code:', 'basr-core' ); ?></label>
				<br />
				<input style="width: 100%" type="text" name="tee_tracking_number" class="tee_tracking_number"
				       placeholder="<?php _e ( 'Enter tracking code', 'basr-core' ); ?>"
				       value="<?php echo esc_attr( $tracking_number ); ?>" />
			</p>
			
			<p>
				<label
					for="order_tracking_link"> <?php _e ( 'Tracking link:', 'basr-core' ); ?></label>
				<br />
				<input style="width: 100%" type="text" class="order_tracking_link" name="order_tracking_link"
				       placeholder="<?php _e ( 'Enter Tracking link', 'basr-core' ); ?>"
				       value="<?php echo esc_attr( $tracking_url ); ?>" />
			</p>
			<div class="line-items">
				<?php 
				$i = 1;
				foreach ( $line_items as $key => $item) {
					$item = (array) $item;
					$sku = isset( $item['sku'] ) ? $item['sku'] : '';
					$quantity = isset( $item['quantity'] ) ? $item['quantity'] : '';
					if( ! empty( $sku ) || ! empty( $quantity ) ) {
						echo '<p class="item">' . $i++ . '. '  . '<span class="item-sku">' . esc_html( $sku ) . '</span>' . ' x <span class="item-quantity">' . esc_html( $quantity ) . '</span></p>';
					}
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Normalize a tracking entry to the correct format
	 * Ensures line_items is an array and all required fields are present
	 */
	private function normalize_tracking_entry( $tracking ) {
		
		if( ! is_array( $tracking ) ) {
			return null;
		}
		
		// Get tracking number from multiple possible fields
		$tracking_num = '';
		if( isset( $tracking['tracking_number'] ) && ! empty( trim( $tracking['tracking_number'] ) ) ) {
			$tracking_num = trim( $tracking['tracking_number'] );
		} elseif( isset( $tracking['tracking'] ) && ! empty( trim( $tracking['tracking'] ) ) ) {
			$tracking_num = trim( $tracking['tracking'] );
		}
		
		// Check if line_items exists and has content (handle both array and object formats)
		$has_line_items = false;
		if( isset( $tracking['line_items'] ) ) {
			if( is_array( $tracking['line_items'] ) ) {
				$has_line_items = count( $tracking['line_items'] ) > 0;
			} elseif( is_object( $tracking['line_items'] ) ) {
				$has_line_items = count( (array) $tracking['line_items'] ) > 0;
			} else {
				$has_line_items = ! empty( $tracking['line_items'] );
			}
		}
		
		// Validate: must have tracking number OR line_items
		$is_valid = false;
		if( ! empty( $tracking_num ) && strlen( $tracking_num ) >= 5 ) {
			$is_valid = true;
		} elseif( $has_line_items ) {
			$is_valid = true;
		}
		
		if( ! $is_valid ) {
			return null;
		}
		
		// Build normalized entry
		$normalized = [];
		
		// Ensure tracking_number field exists
		$normalized['tracking_number'] = ! empty( $tracking_num ) ? $tracking_num : ( isset( $tracking['tracking_number'] ) ? $tracking['tracking_number'] : '' );
		
		// Ensure tracking field matches tracking_number
		$normalized['tracking'] = ! empty( $tracking_num ) ? $tracking_num : ( isset( $tracking['tracking'] ) ? $tracking['tracking'] : $normalized['tracking_number'] );
		
		// Handle tracking_url: preserve if exists and not empty, otherwise use default URL template
		if( isset( $tracking['tracking_url'] ) && ! empty( trim( $tracking['tracking_url'] ) ) && strlen( trim( $tracking['tracking_url'] ) ) > 5 ) {

			$normalized['tracking_url'] = trim( $tracking['tracking_url'] );
		} else {

			// Use default URL template if tracking_url is empty or missing
			$default_url = get_option( 'teeAllover-tracking-url' );
			if( ! empty( $default_url ) && ! empty( $normalized['tracking_number'] ) ) {

				$normalized['tracking_url'] = preg_replace( '/{tracking_number}/', $normalized['tracking_number'], $default_url );
			}
		}
		
		// Normalize line_items: convert object to array and preserve all fields
		$normalized['line_items'] = [];
		if( isset( $tracking['line_items'] ) ) {
			// Convert to array if it's an object
			$line_items = tee_convert_obj_to_array( $tracking['line_items'] );
			if( is_array( $line_items ) ) {
				// Iterate through line_items and normalize each item
				foreach( $line_items as $item_key => $item ) {
					if( is_array( $item ) || is_object( $item ) ) {
						$item = (array) $item;
						// Preserve all fields from the item
						$normalized_item = [];
						if( isset( $item['tracking'] ) ) {
							$normalized_item['tracking'] = $item['tracking'];
						}
						if( isset( $item['url'] ) ) {
							$normalized_item['url'] = $item['url'];
						}
						if( isset( $item['sku'] ) ) {
							$normalized_item['sku'] = $item['sku'];
						}
						if( isset( $item['item_id'] ) ) {
							$normalized_item['item_id'] = $item['item_id'];
						}
						if( isset( $item['custom_sku'] ) ) {
							$normalized_item['custom_sku'] = $item['custom_sku'];
						}
						if( isset( $item['quantity'] ) ) {
							$normalized_item['quantity'] = $item['quantity'];
						}
						if( isset( $item['sync_id'] ) ) {
							$normalized_item['sync_id'] = $item['sync_id'];
						}
						// Only add if item has at least one field
						if( ! empty( $normalized_item ) ) {
							$normalized['line_items'][] = $normalized_item;
						}
					}
				}
			}
		}
		
		return $normalized;
	}

	function save_meta_box( $post_id ) {

		$order = $this->get_cached_order( $post_id );

		if ( isset( $_POST['teeAllover_fulfill'] ) ) {
			// Validate and clean the tracking data before saving
			$fulfill_data = $_POST['teeAllover_fulfill'];
			
			// Initialize as empty array
			$cleaned_data = [];
			
			// If it's a string, try to decode it
			$trimmed = trim( $fulfill_data );
			if( is_string( $fulfill_data ) && ! empty( $trimmed ) && $trimmed !== '{}' && $trimmed !== '[]' ) {
				// Decode the JSON string - WordPress may have added slashes, so try both
				$decoded = json_decode( $fulfill_data, true );
				
				// Check for JSON decode errors
				if( json_last_error() !== JSON_ERROR_NONE ) {
					// If decode failed, try without slashes (WordPress magic quotes)
					$decoded = json_decode( stripslashes( $fulfill_data ), true );
				}
				
				if( $decoded !== null && is_array( $decoded ) ) {
					// If it's an associative array with numeric string keys, convert to indexed array
					if( ! empty( $decoded ) && array_keys( $decoded ) !== range( 0, count( $decoded ) - 1 ) ) {
						$decoded = array_values( $decoded );
					}
					// Normalize each tracking entry
					foreach( $decoded as $key => $tracking ) {
						if( is_array( $tracking ) ) {
							$normalized = $this->normalize_tracking_entry( $tracking );
							if( $normalized !== null ) {
								$cleaned_data[] = $normalized;
							}
						}
					}
				}
			} elseif( is_array( $fulfill_data ) ) {
				// If it's already an array, normalize each entry
				foreach( $fulfill_data as $key => $tracking ) {
					if( is_array( $tracking ) ) {
						$normalized = $this->normalize_tracking_entry( $tracking );
						if( $normalized !== null ) {
							$cleaned_data[] = $normalized;
						}
					}
				}
			}
			
			// Always save as JSON array (not object), even if empty
			// Use JSON_UNESCAPED_UNICODE for proper encoding
			// Slashes will be escaped as \/ by default (which is correct for the format)
			$fulfill_data = json_encode( array_values( $cleaned_data ), JSON_UNESCAPED_UNICODE );
			
			// Save the data (even if empty array, to clear old data)
			if( $fulfill_data !== false ) {
				$this->update_meta( $order ? $order : $post_id, 'teeAllover_fulfill', $fulfill_data );
			}
		}
		if ( isset( $_POST['tee_total_fulfill'] ) ) {
			$this->update_meta( $order ? $order : $post_id, 'tee_total_fulfill', $_POST['tee_total_fulfill'] );
		}
		if ( isset( $_POST['tee_total_items'] ) ) {
			$this->update_meta( $order ? $order : $post_id, 'tee_total_items', $_POST['tee_total_items'] );
		}
		if( isset( $_POST['tee_total_fulfill'] ) && isset( $_POST['tee_total_items'] ) ) {
			$is_fulfill = 'Unfulfilled';
			if( $_POST['tee_total_fulfill'] && $_POST['tee_total_fulfill'] <  $_POST['tee_total_items'] ) {
				$is_fulfill = 'Partially-Fulfilled';
			}
			if( $_POST['tee_total_fulfill'] >=  $_POST['tee_total_items'] ) {
				$is_fulfill = 'Fulfilled';
			}
			$this->update_meta( $order ? $order : $post_id, 'tee_fulfill_status', $is_fulfill );

			if( $is_fulfill == 'Fulfilled' && get_option( 'tee_to_complete' ) == 'true' ) {
				if( $order && is_a( $order, 'WC_Order' ) ) {
					$rs = $order->update_status( 'completed' ); 
				}
			}
		}

		$authen = $this->get_meta( $order ? $order : $post_id, 'teeAllover_order_authen' );

 		if ( ! $authen ) {
 			$authen = 'order-' . $post_id . '-' . md5(uniqid(rand(), true));
 			$this->update_meta( $order ? $order : $post_id, 'teeAllover_order_authen', $authen );
 		}

		// Handle resend tracking email
		if( isset( $_POST['tee-resend-tracking'] ) && $_POST['tee-resend-tracking'] == 'on' ) {
			// sent mail for customer

			// Get order if not already available
			if( ! $order || ! is_a( $order, 'WC_Order' ) ) {
				$order = $this->get_cached_order( $post_id );
			}

			if( ! $order ) return;

			// Use POST data (current state) instead of saved meta to get latest tracking data
			$trackings = null;
			if( isset( $_POST['teeAllover_fulfill'] ) && ! empty( $_POST['teeAllover_fulfill'] ) ) {
				$trackings = json_decode( $_POST['teeAllover_fulfill'], true );
			}
			
			// Fallback to saved meta if POST data is not available
			if( empty( $trackings ) ) {
				$trackings = $this->get_meta( $order, 'teeAllover_fulfill' );
				$trackings = json_decode( $trackings, true );
			}
			
			// Validate trackings data
			if( empty( $trackings ) || ! is_array( $trackings ) ) {
				$order->add_order_note( 'Manual tracking email: ERROR - No valid tracking data found' );
				return;
			}
			
			$trackings = tee_convert_obj_to_array( $trackings ); 

			$M 	 	  	  = Tee_mail::get_instance();
			$template 	  = $M->get_template( 'Update Tracking Number' );

			$billing_mail = $order->get_billing_email();
			$order_number = $order->get_order_number();

			$obj = new StdClass();
			$obj->site_title   = get_bloginfo( 'name' );
			$obj->site_url     = home_url('/');

			$custom_logo_id    = get_theme_mod( 'custom_logo' );
			if( ! $custom_logo_id ) {
				// $logo_img = 'https://pro.teeallover.com/wp-content/uploads/2019/07/tee-allover-600.png';
			} else {
				$logo_img = wp_get_attachment_url( $custom_logo_id );
			}

			$obj->logo         = '<a href="' . $obj->site_url . '"><img src="' . $logo_img . '" alt="logo"></a>';
			$obj->order_number = $order_number;
			$obj->trackings    = '';
			$obj->admin_email  = get_option('admin_email');

			$default_url = get_option( 'teeAllover-tracking-url' );


			$numbers = [];
			foreach( $trackings as $key => $tracking ) {

				// Validate tracking data structure
				if( ! is_array( $tracking ) ) {
					continue;
				}

				// Get tracking number - check multiple possible field names
				$tracking_number = isset( $tracking['tracking_number'] ) ? trim( $tracking['tracking_number'] ) : 
				                  ( isset( $tracking['tracking'] ) ? trim( $tracking['tracking'] ) : '' );

				// Skip if tracking number is missing or too short
				if( empty( $tracking_number ) || strlen( $tracking_number ) < 5 ) {
					continue;
				}

				// Get tracking URL
				$tracking_url = isset( $tracking['tracking_url'] ) ? trim( $tracking['tracking_url'] ) : '';

				// Build tracking link
				if( ! empty( $tracking_url ) && strlen( $tracking_url ) > 5 ) {
					$obj->trackings .= '<a href="' . esc_url( $tracking_url ) . '">' . esc_html( $tracking_number ) . '</a> ,';
				} else {
					// Use default URL template if no URL provided
					if( ! empty( $default_url ) ) {
						$tracking_url = preg_replace('/{tracking_number}/', $tracking_number, $default_url );
						$obj->trackings .= '<a href="' . esc_url( $tracking_url ) . '">' . esc_html( $tracking_number ) . '</a> ,';
					} else {
						// Fallback if no default URL
						$obj->trackings .= esc_html( $tracking_number ) . ' ,';
					}
				}

				$numbers[] = $tracking_number;
			}

			// Validate that we have at least one valid tracking number
			if( empty( $numbers ) ) {
				$order->add_order_note( 'Manual tracking email: ERROR - No valid tracking numbers found to send' );
				return;
			}

			$obj->trackings = preg_replace('/\s,$/', '.', $obj->trackings);

			$obj->order_url = get_permalink( $post_id );

			$rs_sent = $M->sent_mail( 'Update Tracking Number', $billing_mail, $obj );

			$note  = 'Manual Sent number sent ' . ( $rs_sent ? 'success' : 'ERROR' ) . ' to customer: ' . implode( ', ', $numbers );

			$order->add_order_note(
				$note
			);

		}

		return;
		
		if ( $order ) {

			if ( isset( $_POST['tee_tracking_number'] ) ) {
				update_post_meta( $post_id, 'tee_tracking_number', $_POST['tee_tracking_number'] );
			}
			
			if ( isset( $_POST['order_tracking_link'] ) ) {
				update_post_meta( $post_id, 'order_tracking_link', $_POST['order_tracking_link'] );
			}

			$page = get_page_by_title('View Order');
			$view_order_link = '';
	 		if ( is_object( $page ) && isset( $page->ID ) ) {
	 			$view_order_link = get_permalink( $page->ID );
	 		}
	 		$authen = get_post_meta( $post_id, 'tee_shop_order_authen', true );

	 		if ( ! $authen ) {
	 			$authen = uniqid('order-');
	 			update_post_meta( $post_id, 'tee_shop_order_authen', $authen );
	 		}

	 		if ( $view_order_link ) {
	 			$view_order_link = add_query_arg( 'order', $post_id, $view_order_link );
	 			$view_order_link = add_query_arg( 'authen', $authen, $view_order_link );
	 		}

			if ( $_POST['tee_tracking_number'] ) {
				
				$order        = new WC_Order( $post_id );
				$billing_mail = $order->get_billing_email();
				$order_number = $order->get_order_number();
				$M 	 	  	  = Tee_mail::get_instance();
				$template 	  = $M->get_template('sent_tracking');
				if ( ! $template ) return;
				$subject  = $template->post_title . ' #' . $order_number;
				if ( $_POST['order_tracking_link'] ) {
					$link =  $_POST['order_tracking_link'];
				} else {
					$link 	  = $this->link . $_POST['tee_tracking_number'];
				}
				$link = '<a href="' . $link . '">' . $_POST['tee_tracking_number'] . '</a>';

				$store_image = '<img src="' . get_site_icon_url() . '" width="100" alt="logo">';

				$order_url = $view_order_link ? $view_order_link : $order->get_view_order_url();

				$msg 		  = $template->post_content;
				$msg 		  = preg_replace('/{{order\.name}}/', $order_number, $msg);
				$msg 		  = preg_replace('/{{shop\.url}}/', home_url('/'), $msg);
				$msg          = preg_replace('/{{order\.tracking_number}}/', $link, $msg );
				$msg          = preg_replace('/{{order.url}}/', $order_url, $msg );
				$msg          = preg_replace('/{{shop\.img}}/', $store_image, $msg );

				$msg = $M->get_html_format( $msg, 'Shipping Updated!' );

				$email = array( $billing_mail );
				// $email = array( 'haint@byjoomla.com' );

				$rs = wp_mail( $email, $subject, $msg );

			}
		}
	}
}

WC_Tracking::get_instance();

function wpse27856_set_content_type(){
    return "text/html";
}
add_filter( 'wp_mail_content_type','wpse27856_set_content_type' );

/**
 * Helper function to get order meta with HPOS and legacy fallback
 * 
 * @param int|WC_Order $order_id Order ID or WC_Order object
 * @param string $meta_key Meta key to retrieve
 * @param bool $single Whether to return single value (default: true)
 * @return mixed Meta value or empty string if not found
 */
function tee_get_order_meta( $order_id, $meta_key, $single = true ) {

	$tracking = WC_Tracking::get_instance();
	return $tracking->get_meta( $order_id, $meta_key, $single );
}

/**
 * Helper function to update order meta with HPOS and legacy fallback
 * 
 * @param int|WC_Order $order_id Order ID or WC_Order object
 * @param string $meta_key Meta key to update
 * @param mixed $meta_value Meta value to save
 * @return bool|int True on success, meta ID on success, false on failure
 */
function tee_update_order_meta( $order_id, $meta_key, $meta_value ) {

	$tracking = WC_Tracking::get_instance();
	return $tracking->update_meta( $order_id, $meta_key, $meta_value );
}

// Hook for when order is updated (manual or via WooCommerce)
add_action( 'woocommerce_update_order', 'tee_fulfillment_woocommerce_order_update', 10, 1 );

// Hook for when meta is updated directly (API updates)
// Use high priority to ensure it runs after meta is saved
add_action( 'updated_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );

// Also hook into added_post_meta in case API uses add_post_meta instead of update_post_meta
add_action( 'added_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );

// Hook for REST API order updates (WooCommerce REST API)
add_action( 'woocommerce_rest_insert_shop_order_object', 'tee_fulfillment_rest_api_order_updated', 10, 2 );

// Hook for admin page loads to check for pending tracking updates
add_action( 'admin_init', 'tee_fulfillment_check_pending_updates', 999 );

// Periodic check via cron (every 5 minutes)
if( ! wp_next_scheduled( 'tee_fulfillment_periodic_check' ) ) {

	wp_schedule_event( time(), 'tee_fulfillment_interval', 'tee_fulfillment_periodic_check' );
}
add_action( 'tee_fulfillment_periodic_check', 'tee_fulfillment_process_pending_orders' );

// Add custom cron interval
add_filter( 'cron_schedules', function( $schedules ) {

	$schedules['tee_fulfillment_interval'] = [
		'interval' => 300, // 5 minutes
		'display'  => 'Every 5 Minutes'
	];
	return $schedules;
} );

/**
 * Trigger processing when teeAllover_fulfill_2 meta is updated
 */
function tee_fulfillment_meta_updated( $meta_id, $object_id, $meta_key, $meta_value ) {

	// Only process if it's the tracking meta key and it's an order
	if( $meta_key !== 'teeAllover_fulfill_2' ) {

		return;
	}

	// Prevent infinite loop - don't process if we're already processing
	static $processing = [];
	
	if( isset( $processing[$object_id] ) ) {

		return;
	}

	$processing[$object_id] = true;

	// Check if it's a WooCommerce order - use cached version
	$tracking = WC_Tracking::get_instance();
	$order = $tracking->get_cached_order( $object_id );
	
	if( ! $order ) {

		unset( $processing[$object_id] );
		return;
	}

	// Store meta value in global for use in processing function
	if( ! empty( $meta_value ) ) {

		$GLOBALS['_tee_fulfillment_meta_value'] = $meta_value;
	}

	// Process immediately - the function will check for meta in global if not found in DB
	tee_fulfillment_woocommerce_order_update( $object_id );
	
	// Also schedule on shutdown as backup
	add_action( 'shutdown', function() use ( $object_id ) {

		// Double-check the meta exists before processing
		$check_meta = tee_get_order_meta( $object_id, 'teeAllover_fulfill_2' );
		
		if( ! empty( $check_meta ) ) {

			// Check if already processed
			$last_processed = tee_get_order_meta( $object_id, '_tee_fulfillment_last_processed' );
			
			if( $last_processed !== $check_meta ) {

				tee_fulfillment_woocommerce_order_update( $object_id );
			}
		}
	}, 999 );
	
	// Clear processing flag
	unset( $processing[$object_id] );
}

/**
 * Handle REST API order updates
 */
function tee_fulfillment_rest_api_order_updated( $order, $request ) {

	$order_id = $order->get_id();
	
	// Check if meta was updated via REST API
	$meta_data = $request->get_param( 'meta_data' );
	$tracking_value = null;
	
	if( $meta_data && is_array( $meta_data ) ) {

		foreach( $meta_data as $meta ) {

			if( isset( $meta['key'] ) && $meta['key'] === 'teeAllover_fulfill_2' && ! empty( $meta['value'] ) ) {

				// Handle both string and array/object values
				if( is_string( $meta['value'] ) ) {

					$tracking_value = $meta['value'];
				} else {

					$tracking_value = json_encode( $meta['value'] );
				}
				
				break;
			}
		}
	}

	// Also check request body directly (some APIs send it differently)
	if( ! $tracking_value ) {

		$request_body = $request->get_body();
		$request_json = json_decode( $request_body, true );
		
		if( $request_json && isset( $request_json['meta_data'] ) && is_array( $request_json['meta_data'] ) ) {

			foreach( $request_json['meta_data'] as $meta ) {

				if( isset( $meta['key'] ) && $meta['key'] === 'teeAllover_fulfill_2' && ! empty( $meta['value'] ) ) {

					$tracking_value = is_string( $meta['value'] ) ? $meta['value'] : json_encode( $meta['value'] );
					break;
				}
			}
		}
	}

	// If we found the value in the request, store it and process
	if( $tracking_value ) {

		// Store in global for immediate use
		$GLOBALS['_tee_fulfillment_meta_value'] = $tracking_value;
		
		// Also try to get it from order meta (in case it was already saved)
		$order_meta = tee_get_order_meta( $order, 'teeAllover_fulfill_2' );
		
		if( ! empty( $order_meta ) ) {

			$GLOBALS['_tee_fulfillment_meta_value'] = $order_meta;
		}
		
		// Process immediately with the value we captured
		add_action( 'shutdown', function() use ( $order_id, $tracking_value ) {

			tee_fulfillment_woocommerce_order_update( $order_id );
		}, 999 );
	} else {

		// Check if meta exists in database as fallback
		add_action( 'shutdown', function() use ( $order_id ) {

			$meta = tee_get_order_meta( $order_id, 'teeAllover_fulfill_2' );
			
			if( ! empty( $meta ) ) {

				tee_fulfillment_woocommerce_order_update( $order_id );
			}
		}, 999 );
	}
}

/**
 * Check for pending tracking updates on admin page load
 */
function tee_fulfillment_check_pending_updates() {

	// Only run on order edit pages
	global $pagenow;
	
	if( $pagenow !== 'post.php' && ! ( isset( $_GET['page'] ) && $_GET['page'] === 'wc-orders' ) ) {

		return;
	}

	// Get order ID from request
	$order_id = isset( $_GET['post'] ) ? intval( $_GET['post'] ) : ( isset( $_GET['id'] ) ? intval( $_GET['id'] ) : 0 );
	
	if( ! $order_id ) {

		return;
	}

	// Check if there's unprocessed tracking data - use helper function for HPOS compatibility
	$new_trackings = tee_get_order_meta( $order_id, 'teeAllover_fulfill_2' );
	$last_processed = tee_get_order_meta( $order_id, '_tee_fulfillment_last_processed' );
	
	if( ! empty( $new_trackings ) && $new_trackings !== $last_processed ) {

		// Process in background
		add_action( 'shutdown', function() use ( $order_id ) {

			tee_fulfillment_woocommerce_order_update( $order_id );
		}, 999 );
	}
}

/**
 * Periodic check for pending tracking updates
 */
function tee_fulfillment_process_pending_orders() {

	$order_ids = [];
	$limit = 10;
	
	// Find orders with teeAllover_fulfill_2 meta key
	// Use wc_get_orders for both HPOS and legacy compatibility
	$orders = wc_get_orders( [
		'limit'    => $limit * 2, // Get more to filter
		'status'   => 'any',
		'meta_key' => 'teeAllover_fulfill_2',
		'return'   => 'ids',
	] );

	if( ! empty( $orders ) ) {

		$tracking = WC_Tracking::get_instance();
		
		foreach( $orders as $order_id ) {

			$order = $tracking->get_cached_order( $order_id );
			
			if( ! $order ) {
				continue;
			}

			// Get tracking data using helper function
			$new_trackings = tee_get_order_meta( $order, 'teeAllover_fulfill_2' );
			
			// Skip if empty
			if( empty( $new_trackings ) ) {
				continue;
			}

			// Check if already processed
			$last_processed = tee_get_order_meta( $order, '_tee_fulfillment_last_processed' );
			
			// If not processed or data changed, add to processing list
			if( empty( $last_processed ) || $last_processed !== $new_trackings ) {

				$order_ids[] = $order_id;
				
				// Stop when we have enough
				if( count( $order_ids ) >= $limit ) {
					break;
				}
			}
		}
	}

	if( ! empty( $order_ids ) ) {

		foreach( $order_ids as $order_id ) {

			tee_fulfillment_woocommerce_order_update( $order_id );
		}
	}
}

function tee_fulfillment_woocommerce_order_update( $order_id ) {

	// Prevent infinite loops - don't process if we're already processing this order
	static $processing = [];
	
	if( isset( $processing[$order_id] ) ) {

		return;
	}

	$processing[$order_id] = true;

	// Skip if this is a manual edit from admin (but allow processing from meta box display)
	// Only skip if we're in the middle of saving a post edit (POST request with editpost action)
	if( isset( $_POST['action'] ) && $_POST['action'] == 'editpost' && isset( $_POST['save'] ) ) {

		unset( $processing[$order_id] );
		return;
	}

	// Get new trackings from API (stored in teeAllover_fulfill_2)
	// Try multiple methods for HPOS compatibility
	$new_trackings = null;
	
	// Method 1: Check global context first (from REST API request)
	if( isset( $GLOBALS['_tee_fulfillment_meta_value'] ) ) {

		$new_trackings = $GLOBALS['_tee_fulfillment_meta_value'];
		unset( $GLOBALS['_tee_fulfillment_meta_value'] );
	}
	
	// Method 2: Try order meta (HPOS and legacy)
	if( ! $new_trackings ) {

		$new_trackings = tee_get_order_meta( $order_id, 'teeAllover_fulfill_2' );
	}

	// If still empty, schedule a retry on shutdown
	if( ! $new_trackings ) {

		unset( $processing[$order_id] );
		
		// Schedule retry on shutdown
		static $retry_scheduled = [];
		
		if( ! isset( $retry_scheduled[$order_id] ) ) {

			$retry_scheduled[$order_id] = true;
			
			add_action( 'shutdown', function() use ( $order_id ) {

				$retry_meta = tee_get_order_meta( $order_id, 'teeAllover_fulfill_2' );
				
				if( ! empty( $retry_meta ) ) {

					tee_fulfillment_woocommerce_order_update( $order_id );
				}
			}, 999 );
		}
		
		return;
	}

	// Check if this data was already processed (compare with last processed)
	$tracking = WC_Tracking::get_instance();
	$order = $tracking->get_cached_order( $order_id );
	if( ! $order ) {
		unset( $processing[$order_id] );
		return;
	}
	
	$last_processed = tee_get_order_meta( $order, '_tee_fulfillment_last_processed' );
	
	if( $last_processed === $new_trackings ) {

		unset( $processing[$order_id] );
		return;
	}

	$new_trackings = json_decode( $new_trackings, true );

	if( json_last_error() !== JSON_ERROR_NONE ) {

		unset( $processing[$order_id] );
		return;
	}

	// Handle object format from API (tracking number as key)
	// Convert from {"UK610582718YP3":{...}} to [{...}]
	if( ! is_array( $new_trackings ) || ( is_array( $new_trackings ) && ! isset( $new_trackings[0] ) && ! empty( $new_trackings ) ) ) {

		// It's an object, convert to array
		$converted_trackings = [];
		foreach( $new_trackings as $key => $tracking ) {

			if( is_array( $tracking ) ) {

				// Extract tracking number from URL if present (e.g., tracking_url or tracking_numbe)
				$extracted_tracking = null;
				foreach( [ 'tracking_url', 'tracking_numbe', 'tracking_number_url' ] as $url_field ) {
					if( isset( $tracking[$url_field] ) && ! empty( $tracking[$url_field] ) ) {
						// Try to extract tracking number from URL patterns:
						// - "t.17track.net/en#nums=UK610582718YP8" -> extract after nums=
						// - "track.net/en#nums=UK610582718YP8" -> extract after nums=
						// - Or any alphanumeric string that looks like a tracking number (10+ chars)
						if( preg_match( '/nums=([A-Z0-9]{10,})/i', $tracking[$url_field], $matches ) ) {
							$extracted_tracking = $matches[1];
							break;
						} elseif( preg_match( '/([A-Z]{2}[0-9]{9}[A-Z]{1,2}[0-9]?)/i', $tracking[$url_field], $matches ) ) {
							// Pattern like UK610582718YP8 (2 letters, 9 digits, 1-2 letters, optional digit)
							$extracted_tracking = $matches[1];
							break;
						} elseif( preg_match( '/([A-Z0-9]{10,})/i', $tracking[$url_field], $matches ) ) {
							// Fallback: any alphanumeric string 10+ characters
							$extracted_tracking = $matches[1];
							break;
						}
					}
				}

				// Prioritize tracking number from inside the data over the key
				// Priority: tracking_number > tracking > extracted from URL > key
				if( isset( $tracking['tracking_number'] ) && ! empty( $tracking['tracking_number'] ) ) {
					// Use the tracking_number from inside the data (latest/correct one)
					if( ! isset( $tracking['tracking'] ) ) {
						$tracking['tracking'] = $tracking['tracking_number'];
					}
				} elseif( isset( $tracking['tracking'] ) && ! empty( $tracking['tracking'] ) ) {
					// Use tracking field if tracking_number is missing
					$tracking['tracking_number'] = $tracking['tracking'];
				} elseif( $extracted_tracking ) {
					// Use extracted tracking number from URL
					$tracking['tracking_number'] = $extracted_tracking;
					$tracking['tracking'] = $extracted_tracking;
				} else {
					// Fallback to key only if all else fails
					$tracking['tracking_number'] = $key;
					$tracking['tracking'] = $key;
				}
				$converted_trackings[] = $tracking;
			}
		}
		$new_trackings = $converted_trackings;
	}

	// Convert to array format
	$new_trackings = tee_convert_obj_to_array( $new_trackings );

	// Get current trackings using helper function
	$crr_trackings = tee_get_order_meta( $order, 'teeAllover_fulfill' );

	if( ! $crr_trackings ) {

		$crr_trackings = array();
	} else {

		$crr_trackings = json_decode( $crr_trackings, true );
		$crr_trackings = tee_convert_obj_to_array( $crr_trackings );
	}

	$list_number = [];
	$new_number  = [];

	foreach( $crr_trackings as $k => $v ) {
		$list_number[ trim( $v['tracking_number'] ) ] = 1;
	}

	// Get old trackings using helper function
	$old_trackings = tee_get_order_meta( $order, 'teeAllover_fulfill_1' );

	if ( $old_trackings ) {
		$old_trackings = json_decode( $old_trackings );
		$old_trackings = tee_convert_obj_to_array( $old_trackings );

		foreach( $old_trackings as $key => $tracking ) {
			foreach( $crr_trackings as $k => $t ) {
				if ( $old_trackings[$key]['tracking_number'] == $crr_trackings[$k]['tracking_number'] ) {
					unset( $crr_trackings[$k] );
				}
			}
		}
	}
	
	$old_trackings = $new_trackings;
	// Use helper function for HPOS and legacy compatibility
	tee_update_order_meta( $order, 'teeAllover_fulfill_1', json_encode( $old_trackings ) );

	foreach( $new_trackings as $key => $tracking ) {

		// Ensure tracking_number exists - prioritize actual tracking number from data
		if( ! isset( $tracking['tracking_number'] ) || empty( $tracking['tracking_number'] ) ) {

			if( isset( $tracking['tracking'] ) && ! empty( $tracking['tracking'] ) ) {

				$tracking['tracking_number'] = $tracking['tracking'];
			} else {

				continue; // Skip if no tracking number
			}
		}
		
		// Ensure tracking field matches tracking_number (use latest/correct value)
		if( ! isset( $tracking['tracking'] ) || $tracking['tracking'] !== $tracking['tracking_number'] ) {
			$tracking['tracking'] = $tracking['tracking_number'];
		}

		$tracking_number = trim( $tracking['tracking_number'] );

		// Skip if tracking number is too short
		if( strlen( $tracking_number ) < 5 ) {

			continue;
		}

		// Add new tracking to the beginning of array (newest first)
		array_unshift( $crr_trackings, $tracking );
		
		if( ! isset( $list_number[ $tracking_number ] ) ) {

			$new_number[] = $tracking_number;
		}
	}

	// Temporarily remove hooks to prevent loops when updating meta
	remove_action( 'woocommerce_update_order', 'tee_fulfillment_woocommerce_order_update', 10, 1 );
	remove_action( 'updated_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );
	remove_action( 'added_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );

	// Remove duplicates and ensure newest tracking numbers are first
	// Deduplicate by tracking_number, keeping the newest version
	if( ! empty( $crr_trackings ) ) {
		
		$seen_tracking_numbers = [];
		$deduplicated_trackings = [];
		
		// Process in reverse to keep newest when duplicates found
		foreach( array_reverse( $crr_trackings ) as $tracking ) {
			
			$tracking_num = isset( $tracking['tracking_number'] ) ? trim( $tracking['tracking_number'] ) : 
			               ( isset( $tracking['tracking'] ) ? trim( $tracking['tracking'] ) : '' );
			
			if( ! empty( $tracking_num ) && ! isset( $seen_tracking_numbers[$tracking_num] ) ) {
				
				$seen_tracking_numbers[$tracking_num] = true;
				$deduplicated_trackings[] = $tracking;
			}
		}
		
		// Reverse back to show newest first
		$crr_trackings = array_reverse( $deduplicated_trackings );
	}

	// Use helper functions for HPOS and legacy compatibility
	tee_update_order_meta( $order, 'teeAllover_fulfill', json_encode( $crr_trackings ) );
	
	// Mark this data as processed - get the current value first
	$current_fulfill_2 = tee_get_order_meta( $order, 'teeAllover_fulfill_2' );
	tee_update_order_meta( $order, '_tee_fulfillment_last_processed', $current_fulfill_2 );

	// Re-add hooks after meta updates
	add_action( 'woocommerce_update_order', 'tee_fulfillment_woocommerce_order_update', 10, 1 );
	add_action( 'updated_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );
	add_action( 'added_post_meta', 'tee_fulfillment_meta_updated', 999, 4 );

	// SENT MAIL

	$trackings = tee_get_order_meta( $order, 'teeAllover_fulfill' );
	$trackings = json_decode( $trackings, true );
	$trackings = tee_convert_obj_to_array( $trackings );

	tee_update_fulfill_status( $order_id );

	if( count( $new_number ) > 0 ) {

		$M 	 	  	  = Tee_mail::get_instance();

		// Use cached order for HPOS compatibility
		$tracking = WC_Tracking::get_instance();
		$order = $tracking->get_cached_order( $order_id );
		
		if( ! $order ) {

			return; // Can't send email without order
		}

		$billing_mail = $order->get_billing_email();
		$order_number = $order->get_order_number();

		if( empty( $billing_mail ) ) {

			return;
		}

		$obj = new StdClass();
		$obj->site_title   = get_bloginfo( 'name' );
		$obj->site_url     = home_url('/');
		$custom_logo_id    = get_theme_mod( 'custom_logo' );
		if( ! $custom_logo_id ) {
			// $logo_img = 'https://pro.teeallover.com/wp-content/uploads/2019/07/tee-allover-600.png';
		} else {
			$logo_img = wp_get_attachment_url( $custom_logo_id );
		}
		$obj->logo         = '<a href="' . $obj->site_url . '"><img src="' . $logo_img . '" alt="logo"></a>';
		$obj->order_number = $order_number;
		$obj->trackings    = '';
		$obj->admin_email  = get_option('admin_email');

		$default_url = get_option( 'teeAllover-tracking-url' );

		foreach( $trackings as $key => $tracking ) {

			// Handle different tracking number field names
			if( ! isset( $tracking['tracking_number'] ) || empty( $tracking['tracking_number'] ) ) {

				if( isset( $tracking['tracking'] ) && ! empty( $tracking['tracking'] ) ) {

					$trackings[$key]['tracking_number'] = $tracking['tracking'];
				} else {

					continue; // Skip if no tracking number
				}
			}

			$tracking_number = $trackings[$key]['tracking_number'];

			// Skip if tracking number is too short
			if( strlen( $tracking_number ) < 5 ) {

				continue;
			}

			// Use tracking_url if provided, otherwise use default URL with tracking number replacement
			if( isset( $tracking['tracking_url'] ) && ! empty( $tracking['tracking_url'] ) && strlen( $tracking['tracking_url'] ) > 5 ) {

				$tracking_url = $tracking['tracking_url'];
			} else {

				// Use default URL and replace {tracking_number} placeholder
				$tracking_url = $default_url;
				if( $tracking_url ) {

					$tracking_url = preg_replace( '/{tracking_number}/', $tracking_number, $tracking_url );
				} else {

					// Fallback to default link if no default URL is set
					$tracking_url = 'http://parcelsapp.com/en/tracking/' . $tracking_number;
				}
			}

			$obj->trackings .= '<a href="' . esc_url( $tracking_url ) . '">' . esc_html( $tracking_number ) . '</a> ,';
		}
		$obj->trackings = preg_replace( '/\s,$/', '.', $obj->trackings );

		$obj->order_url = get_permalink( $order_id );

		$rs_sent = $M->sent_mail( 'Update Tracking Number', $billing_mail, $obj );

		// Add note to order about email status
		$note  = 'API Tracking update: Tracking number sent ' . ( $rs_sent ? 'success' : 'ERROR' ) . ' to customer: ' . implode( ', ', $new_number );

		$order->add_order_note( $note );

		if( ! $rs_sent ) {

			update_option( 'tee_mail_notices', " Order_id : $order_id - " . $note, 'no' );
		}
	}

	// Clear processing flag at the end of function
	unset( $processing[$order_id] );
}

// Manual trigger via URL: ?tee_process_tracking=ORDER_ID
add_action( 'admin_init', 'tee_fulfillment_manual_trigger' );

/**
 * Manual trigger for processing tracking (for testing/debugging)
 * Usage: /wp-admin/?tee_process_tracking=ORDER_ID
 */
function tee_fulfillment_manual_trigger() {

	if( ! isset( $_GET['tee_process_tracking'] ) || ! current_user_can( 'manage_woocommerce' ) ) {

		return;
	}

	$order_id = intval( $_GET['tee_process_tracking'] );
	
	tee_fulfillment_woocommerce_order_update( $order_id );
	
	wp_die( "Processing triggered for order ID: $order_id.", "Tracking Processing Triggered", [ 'response' => 200 ] );
}

// admin notices for send tracking failed 
add_action( 'admin_notices', 'tee_all_tracking_failed_notice' );

// flow: use global option 'tee_mail_notices', 
function tee_all_tracking_failed_notice( ) {
	$notices = get_option( 'tee_mail_notices' );
	if( ! $notices ) return;
	?>
	<div id="notice-teeAllover" class=" notice notice-error is-dismissible">
		<p><?php echo "TeeAllover Tracking: $notices. Please check your WP mailer logs" ; ?></p>
		<script>
			jQuery(()=>{
				setTimeout( function(){
					jQuery('#notice-teeAllover .notice-dismiss').on( 'click', function(){
						jQuery.ajax({
							url: '<?php echo admin_url('admin-ajax.php'); ?>',
							type: 'GET',
							data: {
								action: 'tee_dismiss_tracking_failed_notice',
								tee_dismiss_tracking_failed_notice: 'true'
							},
							success: function( data ) {
							}
						});
					});
				}, 1000 );
			});
		</script>
	</div>
	<?php
}

add_action( 'wp_ajax_tee_dismiss_tracking_failed_notice', 'tee_dismiss_tracking_failed_notice_ajax' );
add_action( 'wp_ajax_nopriv_tee_dismiss_tracking_failed_notice', 'tee_dismiss_tracking_failed_notice_ajax' );

function tee_dismiss_tracking_failed_notice_ajax() {
	update_option( 'tee_mail_notices', '', 'no' );
	wp_send_json_success( [] );
}